<?php

class LBMetadataExport
{

    /**
     * @var Note
     */
    protected $noteEntity;
    protected $noteName = 'SugarCRm metadata for Logic Builder.';
    protected $filename = 'Metadata.txt';
    protected $file_mime_type = 'text/plain';
    protected $file_ext = 'txt';

    protected $metadata = '';

    public function load()
    {
        try {
            $this->collectMetadata();
            $this->createNoteEntity();
            $this->createFileWithMetadata();
            $this->downloadTheFile();
        } catch(Exception $e) {
            $GLOBALS['log']->fatal('Error occur: ' . $e->getMessage());
            throw $e;
        }
    }

    protected function collectMetadata()
    {
        $rawMetadata = $this->loadMetadata();

        $metadata = array();
        $metadata['user_language'] = $this->getUserPrefLanguage();
        $metadata['sugar_metadata'] = $this->getSugarMetadata($rawMetadata['metadata']['contents']);
        $metadata['sugar_labels'] = $this->getSugarLanguages($rawMetadata['languagePreference']['contents'], $metadata['sugar_metadata']['modules']);
        $metadata = array_merge($metadata, $this->getFlowchartMetadata($rawMetadata));

        $jsonEncoded = json_encode($metadata);
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception('Can\'t convert metadata to json.');
        }
        $base64Encoded = base64_encode($jsonEncoded);
        if ($base64Encoded === false) {
            throw new Exception('Can\'t convert metadata to base64.');
        }
        $this->metadata = $base64Encoded;
    }

    protected function getSugarMetadata($rawMetadata)
    {
        return array(
            'server_info' => $rawMetadata['server_info'] ?? [],
            'relationships' => $this->getSugarRelationshipResponse($rawMetadata['relationships']),
            'modules' => $this->getModuleSugarMetadata($rawMetadata['modules'])
        );
    }

    private function getSugarRelationshipResponse(array $rawRelationships)
    {
        $relationships = [];
        foreach ($rawRelationships as $relationshipName => $rawRelationship) {
            if (!isset($rawRelationship['lhs_module']) || !isset($rawRelationship['rhs_module'])) {
                continue;
            }
            $relationships[$relationshipName] = array(
                'lhs_module' => $rawRelationship['lhs_module'],
                'rhs_module' => $rawRelationship['rhs_module']
            );
        }

        return $relationships;
    }

    protected function getModuleSugarMetadata($rawSugarModules)
    {
        $formattedSugarMetadata = array();
        foreach ($rawSugarModules as $moduleName => $modulesData) {
            $formattedSugarMetadata[$moduleName]['fields'] = $this->getSugarFieldsMetadata($modulesData['fields']);
        }
        return $formattedSugarMetadata;
    }

    protected function getSugarFieldsMetadata($rawFieldsMetadata)
    {
        $fieldsMetadata = [];
        foreach ($rawFieldsMetadata as $fieldName => $fieldMetadata) {
            $fieldName = !empty($fieldMetadata['name']) ? $fieldMetadata['name'] : $fieldName;
            $fieldsMetadata[$fieldName] = $this->getSugarFieldMetadata($fieldMetadata);
        }
        return $fieldsMetadata;
    }

    protected function getSugarFieldMetadata($rawFieldMetadata)
    {
        if (!empty($rawFieldMetadata['type']) && in_array($rawFieldMetadata['type'], array(/*'relate', */'link'/*, 'id'*/))) {
            $fieldMetadata = $rawFieldMetadata;
        } else {
            $fieldMetadata = [];
            $fieldMetadata['name'] = !empty($rawFieldMetadata['name']) ? $rawFieldMetadata['name'] : '';
            $fieldMetadata['vname'] = !empty($rawFieldMetadata['vname']) ? $rawFieldMetadata['vname'] : '';
            $fieldMetadata['type'] = !empty($rawFieldMetadata['type']) ? $rawFieldMetadata['type'] : '';
            if (!empty($rawFieldMetadata['options'])) {
                $fieldMetadata['options'] = $rawFieldMetadata['options'];
            }
        }

        $blacklistedFieldAttributes = array(
            'group', 'group_label',  'comment', 'enable_range_search', 'studio', 'duplicate_on_record_copy', 'readonly',
            'massupdate', 'full_text_search', 'isnull', 'reportable', 'processes', 'duplicate_merge', 'sort_on', 'exportable',
            'rows', 'cols', 'default', 'len', 'merge_filter', 'unified_search', 'audited', 'required', 'hideacl',
            'link_class', 'populate_list', 'workflow', 'rname_exists', 'relate_collection', 'displayParams',
            'mandatory_fetch', 'importable', 'bean_filter_field', 'rhs_key_override', 'subquery', 'db_field', 'function',
            'primary_only', 'rel_fields', 'dbType', 'relationship_fields', 'ignore_role', 'source', 'link_type', 'side'
        );
        foreach ($blacklistedFieldAttributes as $blacklistedFieldAttribute) {
            if (isset($fieldMetadata[$blacklistedFieldAttribute])) {
                unset($fieldMetadata[$blacklistedFieldAttribute]);
            }
        }

        return $fieldMetadata;
    }

    protected function getSugarLanguages($rawLanguages, $modules)
    {
        $decodedLanguages = json_decode($rawLanguages, true);
        return array(
            'app_list_strings' => $decodedLanguages['app_list_strings'],
            'app_strings' => array(),
            'mod_strings' => $this->getModuleLanguages($decodedLanguages, $modules)
        );
    }

    protected function getModuleLanguages($decodedLanguages, $modules)
    {
        $moduleLanguages = array();
        $appStrings = $decodedLanguages['app_strings'];
        foreach ($modules as $moduleName => $moduleData) {
            $moduleLanguages[$moduleName] = array();

            if (empty($moduleData['fields']) || !is_array($moduleData['fields'])) {
                continue;
            }

            $modStrings = $decodedLanguages['mod_strings'][$moduleName];

            foreach ($moduleData['fields'] as $fieldName => $fieldData) {
                $fieldLblKey = array_key_exists('vname', $fieldData) ? $fieldData['vname'] : '';
                if (!empty($modStrings[$fieldLblKey])) {
                    $moduleLanguages[$moduleName][$fieldLblKey] = $modStrings[$fieldLblKey];
                } else if(!empty($appStrings[$fieldLblKey])) {
                    $moduleLanguages[$moduleName][$fieldLblKey] = $appStrings[$fieldLblKey];
                }
            }
        }
        return $moduleLanguages;
    }

    protected function isResponseAvailable($response)
    {
        return $response['status'] == 200 && !empty($response['contents']);
    }

    protected function getFlowchartMetadata($rawMetadata)
    {
        $metadata = array(
            'flowchart_metadata' => array(
                'data' => array(),
                'unavailable' => array()
            )
        );
        if ($this->isResponseAvailable($rawMetadata['emailTemplates'])) {
            $metadata['flowchart_metadata']['data']['emailTemplates'] = $rawMetadata['emailTemplates']['contents'];
        } else {
            $metadata['flowchart_metadata']['unavailable'][] = 'emailTemplates';
        }

        if ($this->isResponseAvailable($rawMetadata['eventNames'])) {
            $metadata['flowchart_metadata']['data']['eventNames'] = $rawMetadata['eventNames']['contents'];
        } else {
            $metadata['flowchart_metadata']['unavailable'][] = 'eventNames';
        }

        if ($this->isResponseAvailable($rawMetadata['flowcharts'])) {
            $metadata['flowchart_metadata']['data']['flowcharts'] = $rawMetadata['flowcharts']['contents'];
        } else {
            $metadata['flowchart_metadata']['unavailable'][] = 'flowcharts';
        }

        if ($this->isResponseAvailable($rawMetadata['services'])) {
            $metadata['flowchart_metadata']['data']['services'] = $rawMetadata['services']['contents'];
        } else {
            $metadata['flowchart_metadata']['unavailable'][] = 'services';
        }

        if ($this->isResponseAvailable($rawMetadata['userActions'])) {
            $metadata['flowchart_metadata']['data']['userActions'] = $rawMetadata['userActions']['contents'];
        } else {
            $metadata['flowchart_metadata']['unavailable'][] = 'userActions';
        }

        if ($this->isResponseAvailable($rawMetadata['userCommands'])) {
            $metadata['flowchart_metadata']['data']['userCommands'] = $rawMetadata['userCommands']['contents'];
        } else {
            $metadata['flowchart_metadata']['unavailable'][] = 'userCommands';
        }

        if ($this->isResponseAvailable($rawMetadata['namedConstants'])) {
            $metadata['flowchart_metadata']['data']['namedConstants'] = $rawMetadata['namedConstants']['contents'];
        } else {
            $metadata['flowchart_metadata']['unavailable'][] = 'namedConstants';
        }


        if ($this->isResponseAvailable($rawMetadata['timelineEventTypes'])) {
            $metadata['flowchart_metadata']['data']['timelineEventTypes'] = $rawMetadata['timelineEventTypes']['contents'];
        } else {
            $metadata['flowchart_metadata']['unavailable'][] = 'timelineEventTypes';
        }

        if ($this->isResponseAvailable($rawMetadata['roles'])) {
            $metadata['flowchart_metadata']['data']['roles'] = $rawMetadata['roles']['contents'];
        } else {
            $metadata['flowchart_metadata']['unavailable'][] = 'roles';
        }


        if ($this->isResponseAvailable($rawMetadata['teams'])) {
            $metadata['flowchart_metadata']['data']['teams'] = $rawMetadata['teams']['contents'];
        } else {
            $metadata['flowchart_metadata']['unavailable'][] = 'teams';
        }

        if ($this->isResponseAvailable($rawMetadata['users'])) {
            $metadata['flowchart_metadata']['data']['users'] = $rawMetadata['users']['contents'];
        } else {
            $metadata['flowchart_metadata']['unavailable'][] = 'users';
        }

        return $metadata;
    }

    protected function createNoteEntity()
    {
        global $current_user;

        $this->noteEntity = BeanFactory::getBean('Notes');
        $this->noteEntity->id = Sugarcrm\Sugarcrm\Util\Uuid::uuid1();
        $this->noteEntity->new_with_id = true;
        $this->noteEntity->name = $this->noteName;
        $this->noteEntity->assigned_user_id = $current_user->id;
        $this->noteEntity->filename = $this->filename;
        $this->noteEntity->file_mime_type = $this->file_mime_type;
        $this->noteEntity->file_ext = $this->file_ext;
        $this->noteEntity->file_size = strlen($this->metadata);
        if (!$this->noteEntity->save()) {
            throw new Exception('Can\'t create Note entity to attach metadata.');
        }
    }

    protected function createFileWithMetadata()
    {
        $fileName = $this->noteEntity->id;
        $fileContent = $this->metadata;
        require_once 'include/upload_file.php';
        $file = new UploadFile();
        $file->set_for_soap($this->noteEntity->id, $fileContent);
        $uploadedFilePath = $this->getUploadedFilePath($file, $fileName);
        if (!$file->final_move($uploadedFilePath)) {
            $this->noteEntity->mark_deleted($this->noteEntity->id);
            throw new Exception('Can\'t create File with metadata.');
        }
    }

    private function getUploadedFilePath(UploadFile $file, string $filename): string
    {
        $uploadFilePath = $file->get_upload_path($filename);
        $uploadStream = new UploadStream();
        $uploadSteamFilePath = $uploadStream->getFSPath($uploadFilePath);
        if (SugarAutoloader::fileExists($uploadSteamFilePath)) {
            return str_replace('upload/', 'upload://', $uploadSteamFilePath);
        }
        return $filename;
    }

    protected function downloadTheFile()
    {
        global $service;
        $api = $service;
        if (empty($api) || !($api instanceof ServiceBase)) {
            SugarAutoLoader::requireWithCustom('custom/include/RestService.php', true);
            if (SugarAutoLoader::existingCustomOne('custom/include/RestService.php') && SugarAutoLoader::customClass('RestService') == 'CustomRestService') {
                $api = new CustomRestService();
            }
            if (empty($api) || !($api instanceof ServiceBase)) {
                $api = new RestService();
            }
        }

        if (!empty($api) && $api instanceof ServiceBase) {
            $api->setHeader('Content-Security-Policy', "script-src 'none'");
            SugarAutoLoader::requireWithCustom('include/download_file.php');
            if (class_exists('CustomDownloadFileApi')) {
                $download = new CustomDownloadFileApi($api);
            }
            if (empty($download) || !($download instanceof DownloadFileApi)) {
                $download = new DownloadFileApi($api);
            }

            if (!empty($download) && $download instanceof DownloadFileApi) {
                $download->getFile($this->noteEntity, 'filename', true);
            }
        }
    }

    protected function loadMetadata()
    {
        $args = array(
            'requests' => array(
                'metadata' => [
                    'url' => '/v10/metadata?type_filter=server_info,modules,ordered_labels,relationships',
                    'method' => 'POST'
                ],
                'languagePreference' => [
                    'url' => '/v10/lang/' . $this->getUserPrefLanguage(),
                    'method' => 'GET'
                ],
                'emailTemplates' => [
                    'url' => '/v10/PX_FlowchartTemplates?fields=id,name,context&max_num=-1',
                    'method' => 'GET'
                ],
                'eventNames' => [
                    'url' => '/v10/PX_FlowchartEvents?fields=id,name,target_module,context&max_num=-1',
                    'method' => 'GET'
                ],
                'flowcharts' => [
                    'url' => '/v10/PX_Flowcharts?fields=id,name&max_num=-1',
                    'method' => 'GET'
                ],
                'services' => [
                    'url' => '/v10/PX_FlowchartServices?fields=id,name,system_name,arguments&max_num=-1',
                    'method' => 'GET'
                ],
                'userActions' => [
                    'url' => '/v10/PX_FlowchartClientActions?fields=id,name,target_module,target_view&max_num=-1',
                    'method' => 'GET'
                ],
                'userCommands' => [
                    'url' => '/v10/PX_FlowchartClientCommands?fields=id,name,target_module,target_view&max_num=-1',
                    'method' => 'GET'
                ],
                'namedConstants' => [
                    'url' => '/v10/PX_FlowchartNamedConstants?fields=id,name,field_type,field_value,constant&max_num=-1',
                    'method' => 'GET'
                ],
                'timelineEventTypes' => [
                    'url' => '/v10/PX_TimelineEvents?fields=id,name,target_module&max_num=-1',
                    'method' => 'GET'
                ],
                'roles' => [
                    'url' => '/v10/ACLRoles?fields=id,name&max_num=-1',
                    'method' => 'GET'
                ],
                'teams' => [
                    'url' => '/v10/Teams?fields=id,name&max_num=-1',
                    'method' => 'GET'
                ],
                'users' => [
                    'url' => '/v10/Users?fields=id,full_name,user_name&max_num=-1',
                    'method' => 'GET'
                ],
                'additionalMeta' => [
                    'url' => '/v10/flowchart/meta',
                    'method' => 'GET'
                ]
            )
        );

        require_once("include/api/RestService.php");
        if (SugarAutoLoader::load('custom/include/RestService.php')) {
            $api = new CustomRestService();
        } else {
            $api = new RestService();
        }
        $api->user = $GLOBALS['current_user'];

        require_once 'include/api/BulkRestService.php';
        require_once 'include/api/BulkRestRequest.php';
        require_once 'include/api/BulkRestResponse.php';
        $restResp = new BulkRestResponse($_SERVER);
        $_GET = array(); $_POST = array();
        foreach ($args['requests'] as $name => $request) {
            if (empty($request['url'])) {
                $GLOBALS['log']->fatal("Bulk Api: URL missing for request $name");
                throw new SugarApiExceptionMissingParameter("Invalid request - URL is missing");
            }
        }
        // check all reqs first so that we don't execute any reqs if one of them is broken
        foreach ($args['requests'] as $name => $request) {
            $restReq = new BulkRestRequest($request);
            $restResp->setRequest($name);
            /**
             * @var $rest RestService
             */
            $rest = new BulkRestService($api);
            $rest->setRequest($restReq);
            $rest->setResponse($restResp);
            $rest->execute();
        }
        return $restResp->getResponses();
    }

    protected function getUserPrefLanguage()
    {
        global $current_user;
        // use their current auth language if it exists
        if (!empty($_SESSION['authenticated_user_language'])) {
            $language = $_SESSION['authenticated_user_language'];
        } elseif (!empty($current_user->preferred_language)) {
            // if current auth language doesn't exist get their preferred lang from the user obj
            $language = $current_user->preferred_language;
        } else {
            // if nothing exists, get the sugar_config default language
            $language = $GLOBALS['sugar_config']['default_language'];
        }

        return $language;
    }

}