<?php

require_once 'include/api/SugarApi.php';

class LBPackageManagementApi extends SugarApi
{
    public const DEFAULT_PACKAGE_STATUS = 'enabled';

    private $packageListInfo;

    private $whiteListedSortableFields = array(
        'id', 'package_name', 'flowchart_name',
        'created_by_name', 'created_by_id', 'created_by_email',
        'date_generated', 'project_name', 'project_id', 'lb_host_name'
    );

    private $sortFlagMapping = array(
        'package_name' => SORT_REGULAR,
        'flowchart_name' => SORT_NATURAL,
        'created_by_name' => SORT_NATURAL,
        'created_by_email' => SORT_NATURAL,
        'project_name' => SORT_NATURAL,
    );

    public function registerApiRest()
    {
        return array(
            'lbPackageList' => array(
                'reqType' => 'GET',
                'path' => array('logicBuilder', 'packages'),
                'pathVars' => array('', ''),
                'method' => 'lbPackageList',
            ),
            'activateLbPackage' => array(
                'reqType' => 'POST',
                'path' => array('logicBuilder', 'packages', 'activate'),
                'pathVars' => array('', '', ''),
                'method' => 'activateLbPackage',
            ),
            'deactivateLbPackage' => array(
                'reqType' => 'POST',
                'path' => array('logicBuilder', 'packages', 'deactivate'),
                'pathVars' => array('', '', ''),
                'method' => 'deactivateLbPackage',
            ),
            'lbPackageDownloadFlowchart' => array(
                'reqType' => 'GET',
                'path' => array('logicBuilder', 'package', 'downloadFlowchart'),
                'pathVars' => array('', '', ''),
                'rawReply' => true,
                'allowDownloadCookie' => true,
                'method' => 'lbPackageDownloadFlowchart',
            ),
        );
    }

    public function lbPackageList(ServiceBase $api, array $args, $acl = 'list')
    {
        $data = array();
        $data['next_offset'] = -1;

        $data['records'] = $this->getLbPackageList($args);

        return $data;
    }

    protected function loadRegisteredPackages(){
        @include('custom/application/Ext/LBRegisteredPackages/lb_registered_packages.ext.php');
        if(!empty($LBRegisteredPackages) && is_array($LBRegisteredPackages)){
            return $LBRegisteredPackages;
        }
        return array();
    }

    protected function getLbPackageList($args){
        $registeredPackages = $this->loadRegisteredPackages();

        $records = array();
        foreach($registeredPackages as $packageId => $packageData){
            $records[] = $this->formatLbPackageRow($packageData);
        }

        $records = $this->sortLBPackage($records, $args);

        return $records;
    }

    protected function sortLBPackage($data, $args){
        if(empty($args) || empty($args['orderBy']))
            return $data;

        if(empty($args['orderBy']['field']) /*|| !in_array($args['orderBy']['field'], $this->whiteListedSortableFields)*/)
            return $data;

        $orderField =  $args['orderBy']['field'];
        $orderDirection =  $args['orderBy']['direction'] == 'asc' ? SORT_ASC : SORT_DESC;


        $sortedData = [];
        foreach ($data as $key => $row) {
            $sortedData[$key]  = $row[$orderField];
        }

        $sortFlag = $this->sortFlagMapping[$orderField] ?? SORT_REGULAR;
        array_multisort($sortedData, $orderDirection, $sortFlag, $data);

        return $data;
    }

    protected function formatLbPackageRow($packageData){
        if(empty($packageData) || !is_array($packageData))
            return array();

        $packageId = $this->getPackageId($packageData);

        return array(
            'id' =>  $packageId,
            'package_name' =>  $this->getPackageName($packageData),
            'flowchart_name' =>  $this->getPackageFlowchartName($packageData),

            'flowchart_version' =>  $this->getPackageFlowchartVersion($packageData),

            'created_by_name' => $this->getPackageCreatedByName($packageData),
            'created_by_id' => $this->getPackageCreatedById($packageData),
            'created_by_email' => $this->getPackageCreatedByEmail($packageData),

            'package_status' => $this->getPackageStatus($packageId),

            'date_generated' => $this->getPackageDateGenerated($packageData),

            'project_name' => $this->getPackageProjectName($packageData),
            'project_id' => $this->getPackageProjectId($packageData),

            'lb_host_name' => $this->getPackageHostName($packageData),

            'is_loadable_flowchart_source' => $this->getIsFlowchartDownloadable($packageId),
            'deactivation_support' => $this->getDeactivationSupport($packageData),
        );
    }

    protected function getPackageId($packageData){
        return $this->getPackageAttributeValue($packageData, array('package_id'));
    }

    protected function getPackageName($packageData){
        return $this->getPackageAttributeValue($packageData, array('package_name'));
    }

    protected function getPackageFlowchartName($packageData){
        return $this->getPackageAttributeValue($packageData, array('package_description'));
    }

    protected function getPackageFlowchartVersion($packageData){
        return $this->getPackageAttributeValue($packageData, array('flowchart_version'), '');
    }

    protected function getPackageCreatedByName($packageData){
        return $this->getPackageAttributeValue($packageData, array('author', 'name'));
    }

    protected function getPackageCreatedByEmail($packageData){
        return $this->getPackageAttributeValue($packageData, array('author', 'email'));
    }

    protected function getPackageStatus($packageId){
        if(!isset($this->packageListInfo)){
            $this->packageListInfo = $this->getPackageListInfo();
        }

        if(!empty($this->packageListInfo[$packageId]) && !empty($this->packageListInfo[$packageId]['status'])){
            return $this->packageListInfo[$packageId]['status'];
        }
        return self::DEFAULT_PACKAGE_STATUS;
    }

    protected function getPackageCreatedById($packageData){
        return $this->getPackageAttributeValue($packageData, array('author', 'id'));
    }

    protected function getPackageDateGenerated($packageData){
        return $this->getPackageAttributeValue($packageData, array('generated_date'));
    }

    protected function getPackageProjectName($packageData){
        return $this->getPackageAttributeValue($packageData, array('project', 'name'));
    }

    protected function getPackageProjectId($packageData){
        return $this->getPackageAttributeValue($packageData, array('project', 'id'));
    }

    protected function getPackageHostName($packageData){
        return $this->getPackageAttributeValue($packageData, array('lb_host', 'name'));
    }

    protected function getIsFlowchartDownloadable($packageId){
        if(empty($packageId)) {
            return false;
        }
        $filePath = $this->getFlowchartSourceFileName($packageId);
        return $this->flowchartSourceFileExists($filePath);
    }

    protected function getDeactivationSupport($packageData){
        return $this->getPackageAttributeValue($packageData, array('deactivation_support'));
    }

    private function getPackageAttributeValue(array $packageData, array $valueKeyPath, $defaultValue = ''){
        $value = $packageData;

        foreach($valueKeyPath as $key){
            if (!isset($value[$key]))
                return $defaultValue;

            $value = $value[$key];
        }

        return $value;
    }

    public function lbPackageDownloadFlowchart(ServiceBase $api, array $args){

        $packageId = !empty($args['package_id']) ? $args['package_id'] : '';
        $flowchartSource = $this->loadFlowchartSource($packageId);
        $fileName = 'flowchart_' . $packageId . '.json';

//        $contentLength = strlen($flowchartSource);

        header("Pragma: public");
        header("Cache-Control: max-age=1, post-check=0, pre-check=0");

        header("Content-Type: application/force-download");
        header("Content-type: application/octet-stream");
        header("Content-Disposition: attachment; filename=\"" . $fileName . "\"");

        header("X-Content-Type-Options: nosniff");
//        header("Content-Length: $contentLength");
        header('Expires: ' . gmdate('D, d M Y H:i:s \G\M\T', time() + 2592000));
        return $flowchartSource;
    }

    protected function loadFlowchartSource($packageId){
        if(empty($packageId)) {
            return '';
        }

        $filePath = $this->getFlowchartSourceFileName($packageId);

        $flowchart = '';
        $flowchartLines = [];
        if($this->flowchartSourceFileExists($filePath)) {
            @include($filePath);
            if (!empty($flowchart) && is_string($flowchart)) {
//                $flowchartDecoded = json_decode($flowchart, true);
//                if(json_last_error() === JSON_ERROR_NONE)
//                    return $flowchartDecoded;
                return base64_decode($flowchart);
            }
            if (!empty($flowchartLines) && is_array($flowchartLines)) {
                return base64_decode(implode('', $flowchartLines));
            }
        }

        throw new SugarApiExceptionNotFound("Flowchart source file could not be found.");
    }

    protected function flowchartSourceFileExists($filePath){
        return SugarAutoLoader::fileExists($filePath);
    }

    protected function getFlowchartSourceFileName($packageId){
        return "custom/PxRoutines/LogicBuilderPackages/".$packageId."/FlowchartSource_".$packageId.".php";
    }

    public function deactivateLbPackage(ServiceBase $api, array $args){
        $packageIds = !empty($args['package_ids']) ? $args['package_ids'] : array();
        $changedIds = $this->changePackageStatus($packageIds, 'disabled');
        return array(
            'package_status' => 'disabled',
            'changed_ids' => $changedIds
        );
    }

    public function activateLbPackage(ServiceBase $api, array $args){
        $packageIds = !empty($args['package_ids']) ? $args['package_ids'] : array();
        $changedIds = $this->changePackageStatus($packageIds, 'enabled');
        return array(
            'package_status' => 'enabled',
            'changed_ids' => $changedIds
        );
    }

    protected function changePackageStatus($packageIds, $newStatus){
        $changedIds = array();

        $registeredPackages = $this->loadRegisteredPackages();
        $packageListInfo = $this->getPackageListInfo($packageIds);
        $db = DBManagerFactory::getInstance();
        foreach($packageIds as $packageId){

            if(empty($registeredPackages[$packageId]) || empty($registeredPackages[$packageId]['deactivation_support']))
                continue;

            $packageInfo = $packageListInfo[$packageId];
            if(!empty($packageInfo) && $packageInfo['status'] == $newStatus){
                $changedIds[] = $packageId;
                continue;
            }

            if(!empty($packageInfo)) {
                $sql = "UPDATE px_lb_package_info SET status='" . $newStatus . "' where package_id='" . $packageId . "'";
            }else{
                $sql = "INSERT INTO px_lb_package_info(id, package_id, status) VALUES('" . Sugarcrm\Sugarcrm\Util\Uuid::uuid1() . "', '" . $packageId . "', '" . $newStatus . "')";
            }
            $db->query($sql);
            $changedIds[] = $packageId;
        }

        return $changedIds;
    }

    protected function getPackageListInfo($packageIds = array()){
        $packageList = array();
        $db = DBManagerFactory::getInstance();
        $sql = "SELECT * from px_lb_package_info";
        if(!empty($packageIds))
            $sql .= " where package_id in ('" . implode("', '", $packageIds) . "')";
        $rs = $db->query($sql);
        while (($row = $db->fetchByAssoc($rs)) != null) {
            $packageList[$row['package_id']] = $row;
        }

        return $packageList;
    }

}